/*
 * Jupiter.cpp
 *
 * Part of Fly! Legacy project
 *
 * Copyright 2003 Chris Wallace
 *
 * Fly! Legacy is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 * Fly! Legacy is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 *   along with Fly! Legacy; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*! \file Jupiter.cpp
 *  \brief Implements CJupiter ephemeris calculator for the planet Jupiter
 *
 *  Implements CJupiter ephemeris calculator for the planet Jupiter.
 *    This code was derived from similar code in SimGear 0.3.3
 *
 */

#include <math.h>
#include "../Include/Ephemeris.h"


//
// Default orbital elements
//
static SOrbitalElements first =
{
  100.4542,   // N, Longitude of the ascending node
  1.3030,     // i, Inclination to the ecliptic
  273.8777,   // w, Argument of perihelion
  5.2025600,    // a, Semi-major axis
  0.048498,   // e, Eccentricity
  19.89500    // M, Mean anomaly
};

static SOrbitalElements second =
{
  2.7685400E-5, // N, Longitude of the ascending node
  -1.557E-7,    // i, Inclination to the ecliptic
  1.6450500E-5, // w, Argument of perihelion
  0.0,      // a, Semi-major axis
  4.469E-9,   // e, Eccentricity
  0.08308530010 // M, Mean anomaly
};


/*************************************************************************
 * CJupiter::CJupiter(double mjd)
 * Public constructor for class CJupiter
 * Argument: The current time.
 * the hard coded orbital elements for CJupiter are passed to 
 * CelestialBody::CelestialBody();
 ************************************************************************/
CJupiter::CJupiter(double mjd) :
  CCelestialBody (first.N, first.i, first.w, first.a, first.e, first.M,
                second.N, second.i, second.w, second.a, second.e, second.M,
          mjd)
{
}

CJupiter::CJupiter() :
  CCelestialBody (first.N, first.i, first.w, first.a, first.e, first.M,
                second.N, second.i, second.w, second.a, second.e, second.M)
{
}


/*************************************************************************
 * void CJupiter::updatePosition(double mjd, CSol *ourSun)
 * 
 * calculates the current position of CJupiter, by calling the base class,
 * CelestialBody::updatePosition(); The current magnitude is calculated using 
 * a Jupiter specific equation
 *************************************************************************/
void CJupiter::UpdatePosition(double mjd, CSol *ourSun)
{
  CCelestialBody::UpdatePosition(mjd, ourSun);
  magnitude = -9.25 + 5*log10( r*R ) + 0.014 * FV;
}


/**************************************************************************
 * jupiter.cxx
 * Written by Durk Talsma. Originally started October 1997, for distribution  
 * with the FlightGear project. Version 2 was written in August and 
 * September 1998. This code is based upon algorithms and data kindly 
 * provided by Mr. Paul Schlyter. (pausch@saaf.se). 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 **************************************************************************/

